// Berkcan Agar.io GL v3 - leaderboard + minimap
const canvas = document.getElementById('game');
const ctx = canvas.getContext('2d');

const joinScreen = document.getElementById('join-screen');
const startBtn = document.getElementById('startBtn');
const playerNameInput = document.getElementById('playerNameInput');

const hud = document.getElementById('hud');
const info = document.getElementById('info');
const hudName = document.getElementById('hudName');
const hudScore = document.getElementById('hudScore');
const hudSize = document.getElementById('hudSize');
const hudPlayers = document.getElementById('hudPlayers');
const hudPos = document.getElementById('hudPos');

const leaderboard = document.getElementById('leaderboard');
const lbList = document.getElementById('lbList');

const chatBox = document.getElementById('chat-box');
const chatMessages = document.getElementById('chat-messages');
const chatInput = document.getElementById('chatInput');
const chatSendBtn = document.getElementById('chatSendBtn');

// Dünya
const WORLD_SIZE = 4000;
const FOOD_COUNT = 220;
const BOT_COUNT  = 5;

let player = null;
let foods = [];
let bots  = [];
let otherPlayers = [];
let camera = {x:0,y:0};
let target = {x:0,y:0};
let gameStarted = false;
let lastChatId = 0;

// Minimap ayarları
const MINI_MARGIN = 12;
const MINI_SIZE = 150;

// Benzersiz id
let myId = localStorage.getItem('agario_gl_player_id_v3');
if(!myId){
  myId = 'p_' + Math.random().toString(36).slice(2);
  localStorage.setItem('agario_gl_player_id_v3', myId);
}

function resizeCanvas(){
  canvas.width = window.innerWidth;
  canvas.height = window.innerHeight;
}
window.addEventListener('resize', resizeCanvas);
resizeCanvas();

function rand(min,max){return Math.random()*(max-min)+min;}
function randomColor(){
  const colors=['#22c55e','#38bdf8','#a855f7','#f97316','#ef4444','#eab308'];
  return colors[Math.floor(Math.random()*colors.length)];
}

function createFoods(){
  foods=[];
  for(let i=0;i<FOOD_COUNT;i++){
    foods.push({
      x:rand(0,WORLD_SIZE),
      y:rand(0,WORLD_SIZE),
      r:rand(3,6),
      c:`hsl(${Math.floor(rand(0,360))},80%,60%)`
    });
  }
}

function createBots(){
  bots=[];
  for(let i=0;i<BOT_COUNT;i++){
    bots.push({
      x:rand(0,WORLD_SIZE),
      y:rand(0,WORLD_SIZE),
      r:rand(12,34),
      c:'#ef4444', // kırmızı
      angle:rand(0,Math.PI*2),
      speed:rand(1.3,2.4)
    });
  }
}

function createPlayer(name){
  const startX = WORLD_SIZE/2+rand(-200,200);
  const startY = WORLD_SIZE/2+rand(-200,200);
  player={
    id:myId,
    name,
    x:startX,
    y:startY,
    r:20,
    baseSpeed:4,
    score:0,
    color:'#22c55e' // yeşil
  };
  target.x=startX;
  target.y=startY;
}

function getPosFromEvent(e){
  let x,y;
  if(e.touches && e.touches[0]){
    x=e.touches[0].clientX;
    y=e.touches[0].clientY;
  }else{
    x=e.clientX; y=e.clientY;
  }
  return {x,y};
}

function handleMove(e){
  if(!gameStarted || !player) return;
  e.preventDefault();
  const pos = getPosFromEvent(e);
  target.x = pos.x - camera.x;
  target.y = pos.y - camera.y;
}

canvas.addEventListener('touchstart',handleMove,{passive:false});
canvas.addEventListener('touchmove',handleMove,{passive:false});
canvas.addEventListener('mousemove',handleMove);

function circlesCollide(a,b){
  const dx=a.x-b.x, dy=a.y-b.y;
  return Math.hypot(dx,dy) < (a.r+b.r);
}

function updatePlayer(){
  if(!player) return;
  const dx=target.x-player.x;
  const dy=target.y-player.y;
  const dist=Math.hypot(dx,dy)||1;
  const speedFactor=30/player.r;
  const speed=player.baseSpeed*Math.max(0.55,Math.min(speedFactor,1.4));
  if(dist>1){
    player.x+=(dx/dist)*speed;
    player.y+=(dy/dist)*speed;
  }
  player.x=Math.max(player.r,Math.min(WORLD_SIZE-player.r,player.x));
  player.y=Math.max(player.r,Math.min(WORLD_SIZE-player.r,player.y));
}

function updateBots(){
  if(!player) return;
  for(const bot of bots){
    const dx=player.x-bot.x, dy=player.y-bot.y;
    const dist=Math.hypot(dx,dy)||1;
    if(dist<600){
      if(bot.r<player.r*0.9){
        bot.x-=(dx/dist)*bot.speed;
        bot.y-=(dy/dist)*bot.speed;
      }else if(bot.r>player.r*1.2){
        bot.x+=(dx/dist)*bot.speed;
        bot.y+=(dy/dist)*bot.speed;
      }else{
        bot.angle+=rand(-0.12,0.12);
        bot.x+=Math.cos(bot.angle)*bot.speed;
        bot.y+=Math.sin(bot.angle)*bot.speed;
      }
    }else{
      if(Math.random()<0.02) bot.angle=rand(0,Math.PI*2);
      bot.x+=Math.cos(bot.angle)*bot.speed;
      bot.y+=Math.sin(bot.angle)*bot.speed;
    }
    bot.x=Math.max(bot.r,Math.min(WORLD_SIZE-bot.r,bot.x));
    bot.y=Math.max(bot.r,Math.min(WORLD_SIZE-bot.r,bot.y));
  }
}

function checkFoodCollision(){
  if(!player) return;
  for(let i=foods.length-1;i>=0;i--){
    if(circlesCollide({x:player.x,y:player.y,r:player.r},foods[i])){
      player.r+=0.25;
      player.score+=4;
      foods[i]={
        x:rand(0,WORLD_SIZE),
        y:rand(0,WORLD_SIZE),
        r:rand(3,6),
        c:`hsl(${Math.floor(rand(0,360))},80%,60%)`
      };
    }
  }
}

function checkBotCollision(){
  if(!player) return;
  for(let i=bots.length-1;i>=0;i--){
    const b=bots[i];
    if(!circlesCollide({x:player.x,y:player.y,r:player.r},b)) continue;
    if(player.r>b.r*1.15){
      player.score+=b.r*10;
      player.r+=b.r*0.25;
      bots[i]={
        x:rand(0,WORLD_SIZE),y:rand(0,WORLD_SIZE),
        r:rand(12,34),
        c:'#ef4444',
        angle:rand(0,Math.PI*2),
        speed:rand(1.3,2.4)
      };
    }else if(b.r>player.r*1.15){
      player.score=Math.max(0,player.score-60);
      player.r=Math.max(18,player.r*0.7);
      player.x=WORLD_SIZE/2+rand(-300,300);
      player.y=WORLD_SIZE/2+rand(-300,300);
    }
  }
}

function checkPlayersEat(){
  if(!player) return;
  for(const op of otherPlayers){
    const a={x:player.x,y:player.y,r:player.r};
    const b={x:op.x,y:op.y,r:op.r};
    if(!circlesCollide(a,b)) continue;
    if(player.r>op.r*1.15){
      player.score+=op.r*12;
      player.r+=op.r*0.2;
    }else if(op.r>player.r*1.15){
      player.score=Math.max(0,player.score-80);
      player.r=Math.max(18,player.r*0.6);
      player.x=WORLD_SIZE/2+rand(-300,300);
      player.y=WORLD_SIZE/2+rand(-300,300);
    }
  }
}

function updateCamera(){
  if(!player) return;
  camera.x=canvas.width/2-player.x;
  camera.y=canvas.height/2-player.y;
}

function drawBackground(){
  ctx.fillStyle='#020617';
  ctx.fillRect(0,0,canvas.width,canvas.height);
  ctx.strokeStyle='rgba(148,163,184,0.18)';
  ctx.lineWidth=1;
  const step=120;
  const startX=Math.floor((-camera.x)/step)*step;
  const endX=startX+canvas.width+step;
  const startY=Math.floor((-camera.y)/step)*step;
  const endY=startY+canvas.height+step;
  for(let x=startX;x<endX;x+=step){
    ctx.beginPath();
    ctx.moveTo(x+camera.x,0);
    ctx.lineTo(x+camera.x,canvas.height);
    ctx.stroke();
  }
  for(let y=startY;y<endY;y+=step){
    ctx.beginPath();
    ctx.moveTo(0,y+camera.y);
    ctx.lineTo(canvas.width,y+camera.y);
    ctx.stroke();
  }
}

function drawFoods(){
  for(const f of foods){
    const sx=f.x+camera.x, sy=f.y+camera.y;
    if(sx<-20||sx>canvas.width+20||sy<-20||sy>canvas.height+20) continue;
    ctx.beginPath();
    ctx.fillStyle=f.c;
    ctx.arc(sx,sy,f.r,0,Math.PI*2);
    ctx.fill();
  }
}

function drawBots(){
  ctx.lineWidth=2;
  for(const b of bots){
    const sx=b.x+camera.x, sy=b.y+camera.y;
    if(sx<-60||sx>canvas.width+60||sy<-60||sy>canvas.height+60) continue;
    ctx.beginPath();
    ctx.fillStyle=b.c;
    ctx.arc(sx,sy,b.r,0,Math.PI*2);
    ctx.fill();
    ctx.strokeStyle='rgba(248,250,252,0.8)';
    ctx.stroke();
    ctx.fillStyle='#e5e7eb';
    ctx.font='11px system-ui';
    ctx.textAlign='center';
    ctx.fillText('BOT',sx,sy-b.r-4);
  }
}

function drawPlayers(){
  for(const op of otherPlayers){
    const sx=op.x+camera.x, sy=op.y+camera.y;
    if(sx<-80||sx>canvas.width+80||sy<-80||sy>canvas.height+80) continue;
    ctx.beginPath();
    ctx.fillStyle=op.color||'#3b82f6'; // mavi
    ctx.arc(sx,sy,op.r,0,Math.PI*2);
    ctx.fill();
    ctx.strokeStyle='rgba(191,219,254,0.9)';
    ctx.lineWidth=3;
    ctx.stroke();
    ctx.fillStyle='#e5e7eb';
    ctx.font='12px system-ui';
    ctx.textAlign='center';
    ctx.fillText(op.name||'Oyuncu',sx,sy-op.r-6);
  }
  if(player){
    const sx=player.x+camera.x, sy=player.y+camera.y;
    ctx.beginPath();
    ctx.fillStyle=player.color; // yeşil
    ctx.arc(sx,sy,player.r,0,Math.PI*2);
    ctx.fill();
    ctx.strokeStyle='#bbf7d0';
    ctx.lineWidth=4;
    ctx.stroke();
    ctx.fillStyle='#e5e7eb';
    ctx.font='14px system-ui';
    ctx.textAlign='center';
    ctx.fillText(player.name,sx,sy-player.r-8);
  }
}

// Minimap
function drawMinimap(){
  const size = MINI_SIZE;
  const x = canvas.width - size - MINI_MARGIN;
  const y = canvas.height - size - MINI_MARGIN;
  const scale = size / WORLD_SIZE;

  // Arka plan kare
  ctx.save();
  ctx.fillStyle='rgba(15,23,42,0.85)';
  ctx.strokeStyle='rgba(148,163,184,0.9)';
  ctx.lineWidth=2;
  ctx.beginPath();
  ctx.rect(x,y,size,size);
  ctx.fill();
  ctx.stroke();

  // Grid çizgileri (koordinat hissi)
  ctx.strokeStyle='rgba(148,163,184,0.25)';
  ctx.lineWidth=1;
  const gridStep = size/4;
  for(let gx=x+gridStep; gx<x+size; gx+=gridStep){
    ctx.beginPath();
    ctx.moveTo(gx,y);
    ctx.lineTo(gx,y+size);
    ctx.stroke();
  }
  for(let gy=y+gridStep; gy<y+size; gy+=gridStep){
    ctx.beginPath();
    ctx.moveTo(x,gy);
    ctx.lineTo(x+size,gy);
    ctx.stroke();
  }

  // Botlar (kırmızı)
  for(const b of bots){
    const mx = x + b.x*scale;
    const my = y + b.y*scale;
    ctx.beginPath();
    ctx.fillStyle='#ef4444';
    ctx.arc(mx,my,3,0,Math.PI*2);
    ctx.fill();
  }

  // Diğer oyuncular (mavi)
  for(const op of otherPlayers){
    const mx = x + op.x*scale;
    const my = y + op.y*scale;
    ctx.beginPath();
    ctx.fillStyle='#3b82f6';
    ctx.arc(mx,my,3.5,0,Math.PI*2);
    ctx.fill();
  }

  // Biz (yeşil)
  if(player){
    const mx = x + player.x*scale;
    const my = y + player.y*scale;
    ctx.beginPath();
    ctx.fillStyle='#22c55e';
    ctx.arc(mx,my,4.5,0,Math.PI*2);
    ctx.fill();
  }

  ctx.restore();
}

// HUD
function updateHud(){
  if(!player) return;
  hudName.textContent=player.name;
  hudScore.textContent=Math.floor(player.score);
  hudSize.textContent=Math.floor(player.r);
  hudPlayers.textContent=otherPlayers.length+1;
  hudPos.textContent = Math.floor(player.x) + ', ' + Math.floor(player.y);
}

// Leaderboard
function updateLeaderboard(){
  if(!player) return;
  const all = otherPlayers.map(p=>({...p,isMe:false}));
  all.push({...player,isMe:true});
  all.sort((a,b)=>b.score-a.score);
  const top5 = all.slice(0,5);
  lbList.innerHTML='';
  top5.forEach((p,idx)=>{
    const row=document.createElement('div');
    row.className='lb-row';
    const nameSpan=document.createElement('span');
    nameSpan.className='lb-name';
    nameSpan.textContent=(idx+1)+'. '+(p.isMe ? '(Sen) '+p.name : p.name);
    const scoreSpan=document.createElement('span');
    scoreSpan.className='lb-score';
    scoreSpan.textContent=Math.floor(p.score);
    row.appendChild(nameSpan);
    row.appendChild(scoreSpan);
    lbList.appendChild(row);
  });
}

// ---- PHP API ----
async function sendUpdateToServer(){
  if(!player) return;
  try{
    const body=new URLSearchParams({
      id:player.id,name:player.name,color:player.color,
      x:String(player.x),y:String(player.y),
      r:String(player.r),score:String(player.score)
    });
    await fetch('api.php?action=update',{method:'POST',body});
  }catch(e){}
}
async function fetchPlayersFromServer(){
  try{
    const res=await fetch('api.php?action=list');
    const data=await res.json();
    if(data && Array.isArray(data.players)){
      otherPlayers=data.players.filter(p=>p.id!==myId);
    }
  }catch(e){}
}
function appendChatMessage(m){
  const div=document.createElement('div');
  div.className='chat-msg';
  const ns=document.createElement('span');
  ns.className='name'; ns.textContent=m.name||'Oyuncu';
  const ts=document.createElement('span');
  ts.className='time'; ts.textContent=m.time||'';
  const tx=document.createElement('span');
  tx.className='text'; tx.textContent=': '+(m.text||'');
  div.appendChild(ns); div.appendChild(ts); div.appendChild(tx);
  chatMessages.appendChild(div);
  chatMessages.scrollTop=chatMessages.scrollHeight;
}
async function fetchChat(){
  try{
    const res=await fetch('api.php?action=chat_list&since='+lastChatId);
    const data=await res.json();
    if(data && Array.isArray(data.messages)){
      for(const m of data.messages){
        if(m.id>lastChatId){
          lastChatId=m.id;
          appendChatMessage(m);
        }
      }
    }
  }catch(e){}
}
async function sendChatMessage(){
  const text=chatInput.value.trim();
  if(!text||!player) return;
  chatInput.value='';
  try{
    const body=new URLSearchParams({name:player.name,text});
    await fetch('api.php?action=chat_send',{method:'POST',body});
  }catch(e){}
}

// Oyun döngüsü
function loop(){
  if(gameStarted && player){
    updatePlayer();
    updateBots();
    checkFoodCollision();
    checkBotCollision();
    checkPlayersEat();
    updateCamera();
    updateHud();
    updateLeaderboard();
  }
  drawBackground();
  if(gameStarted){
    drawFoods();
    drawBots();
    drawPlayers();
    drawMinimap();
  }
  requestAnimationFrame(loop);
}

// Başlat
startBtn.addEventListener('click',()=>{
  let name=playerNameInput.value.trim();
  if(!name) name='Berkcan';
  createFoods();
  createBots();
  createPlayer(name);
  joinScreen.classList.add('hidden');
  hud.classList.remove('hidden');
  info.classList.remove('hidden');
  chatBox.classList.remove('hidden');
  leaderboard.classList.remove('hidden');
  gameStarted=true;
  setInterval(()=>{
    sendUpdateToServer();
    fetchPlayersFromServer();
    fetchChat();
  },450);
});

chatSendBtn.addEventListener('click',sendChatMessage);
chatInput.addEventListener('keydown',e=>{
  if(e.key==='Enter'){
    e.preventDefault();
    sendChatMessage();
  }
});

loop();
